// background.js (Firefox)

browser.runtime.onMessage.addListener((msg, sender) => {
  if (!msg || msg.action !== "downloadFile") return;

  // Must return a Promise or true for async handling in MV2
  return (async () => {
    try {
      const { filename, mime, data, saveAs } = msg;

      if (!filename || !data) {
        throw new Error("Missing filename or data");
      }

      // data is ArrayBuffer
      const blob = new Blob([data], { type: mime || "application/octet-stream" });
      const url = URL.createObjectURL(blob);

      let downloadId;
      try {
        downloadId = await browser.downloads.download({
          url,
          filename,
          saveAs: saveAs !== false, // default true
          conflictAction: "uniquify"
        });
      } catch (e) {
        // Fallback: try without saveAs (some setups block save dialog)
        downloadId = await browser.downloads.download({
          url,
          filename,
          saveAs: false,
          conflictAction: "uniquify"
        });
      }

      const cleanup = (delta) => {
        if (delta.id !== downloadId) return;
        if (delta.state && (delta.state.current === "complete" || delta.state.current === "interrupted")) {
          browser.downloads.onChanged.removeListener(cleanup);
          URL.revokeObjectURL(url);
        }
      };

      browser.downloads.onChanged.addListener(cleanup);

      return { ok: true, downloadId };
    } catch (err) {
      return { ok: false, error: err && err.message ? err.message : String(err) };
    }
  })();
});
